#include        "gpib.h"
#include        "formatio.h"
#include        "tsp3220.h"

/* Thurlby-Thandar TSP3220 series power supplies instrument driver            */
/* LabWindows 2.0 Multi-Instrument Driver                                  */
/* Original Release: Feb 1990                                              */
/* By: John Tothill, Thurlby Thandar Instruments Ltd.                      */
/* Originally written in C                                                 */
/* Modification History: V2.0 updated to latest standard and               */
/*                       and multi-instrument Apr 1995                     */
/*=========================================================================*/

/*= INSTRUMENT TABLE ======================================================*/
/* address array: contains the GPIB addresses of opened instruments.       */
/* bd array: contains the device descriptors returned by OpenDev.          */
/* instr_cnt: contains the number of instruments open of this model type.  */
/* tsp3220_err: the error variable for the instrument module                */
/*=========================================================================*/
static int address[tsp3220_MAX_INSTR + 1];
static int bd[tsp3220_MAX_INSTR + 1];
static int instr_cnt;
static int tsp3220_err;
/*= INTERNAL DATA =========================================================*/
/* 'buffer' is a buffer for GPIB I/O strings */
static char buffer[100];
static long tsp3220_model;

/*= UTILITY ROUTINES ======================================================*/
int tsp3220_open_instr (int);
int tsp3220_close_instr (int);
int tsp3220_invalid_integer_range (int, int, int, int);
int tsp3220_device_closed (int);
int tsp3220_read_data (int, char *, int);
int tsp3220_write_data (int, char *, int);
int tsp3220_read_data_file (int, char *);
int tsp3220_write_data_file (int, char *);
/*=========================================================================*/
/* Function: Initialize                                                    */
/* Purpose:  This function opens the instrument                            */
/*=========================================================================*/
int tsp3220_init (addr,set_default,self_test, instrID)
int addr;
int set_default;
int self_test;
int *instrID;
{
int  ID;
int ident;
    ident = 0;
    if (tsp3220_invalid_integer_range (addr, 0, 30, -1) != 0)
        return tsp3220_err;

    tsp3220_err = 0;
    ID = tsp3220_open_instr (addr);
    if (ID <= 0)
        return tsp3220_err;

    ibtmo (bd[ID], 14);   /* 30 sec timeout */
    if (ibsta & 0x8000)
        {
        tsp3220_err = 300;
        return tsp3220_err;
        }
    ibeot (bd[ID], 1);    /* enable end message */
    if (ibsta & 0x8000)
        {
        tsp3220_err = 300;
        return tsp3220_err;
        }

    ibclr (bd[ID]);
    if (tsp3220_write_data(ID,"*CLS\012",5))  /* clear status */
        return tsp3220_err;
    if (tsp3220_write_data(ID,"*IDN?\012",5))  /* get identifier string */
        return tsp3220_err;
    if (tsp3220_read_data(ID,buffer,50))
        return tsp3220_err;
    Scan(buffer,"THURLBY-THANDAR,TSP%d,0,0,00",&ident); /* check idn */
    if (ident >= 3221 && ident <= 3224)
        tsp3220_model = ident;
    else     /* error unknown id */
        {
        tsp3220_err = 223;
        return tsp3220_err;
       }
    if (set_default)
        if (tsp3220_write_data(ID,"*RST\012",5))
            return tsp3220_err;
    if (self_test)
        {
        if (tsp3220_write_data(ID,"*TST?\012",6))
            return tsp3220_err;
        if (tsp3220_read_data(ID,buffer,50))
            return tsp3220_err;
        Scan(buffer,"%d",&ident);
        if (ident)
            tsp3220_err = ident + 300;
        else
            tsp3220_err = 0;
        }
    *instrID = ID;
    return tsp3220_err;
}
/*=========================================================================*/
/* Function: Set Mode                                                      */
/* Purpose:  Set the mode of operation                                     */
/*=========================================================================*/
int tsp3220_set_mode (instrID,modeb,tracka,trackb,sdm)
int instrID;
int modeb;
int tracka;
int trackb;
int sdm;
{
    if (tsp3220_invalid_integer_range (instrID, 1, tsp3220_MAX_INSTR, -1) != 0)
        return tsp3220_err;
    if (tsp3220_invalid_integer_range (modeb, 0, 12, -2) != 0)
        return tsp3220_err;
    if (tsp3220_invalid_integer_range (tracka, 0, 14, -3) != 0)
        return tsp3220_err;
    if (tsp3220_invalid_integer_range (trackb, 0, 14, -4) != 0)
        return tsp3220_err;
    if (tsp3220_invalid_integer_range (sdm, 0, 1, -5) != 0)
        return tsp3220_err;
    if (tsp3220_device_closed (instrID))
        return tsp3220_err;

    Fmt(buffer,"MODE %d,%d,%d;TRIPMODE %d\012",modeb,tracka,trackb,sdm);
    if (tsp3220_write_data(instrID,buffer,NumFmtdBytes()))
        return tsp3220_err;
    return tsp3220_err;  /* success */
}
/*=========================================================================*/
/* Function: Set volts                                                     */
/* Purpose:  Set the voltage on the sprcified output                       */
/*=========================================================================*/
int tsp3220_set_volts(instrID,opno,voltage)
int instrID;
int opno;
double voltage;
{
    if (tsp3220_invalid_integer_range (instrID, 1, tsp3220_MAX_INSTR, -1) != 0)
        return tsp3220_err;
    tsp3220_err = -2;       /* assume error */
    if (opno < 1)
        return tsp3220_err;
    if (tsp3220_model == 3222 && opno > 2)
        return tsp3220_err;
    tsp3220_err = 0;
    if (tsp3220_device_closed (instrID))
        return tsp3220_err;

    Fmt(buffer,"V%d %f\012",opno,voltage);
    if (tsp3220_write_data(instrID,buffer,NumFmtdBytes()))
        return tsp3220_err;
    return tsp3220_err;  /* success */
}
/*=========================================================================*/
/* Function: Set current                                                   */
/* Purpose:  Set the current limit on the selected output                  */
/*=========================================================================*/
int tsp3220_set_current(instrID,opno, current)
int instrID;
int opno;
double current;
{
    if (tsp3220_invalid_integer_range (instrID, 1, tsp3220_MAX_INSTR, -1) != 0)
        return tsp3220_err;
    tsp3220_err = -2;       /* assume error */
    if (opno < 1)
        return tsp3220_err;
    if (tsp3220_model == 3222 && opno > 2)
        return tsp3220_err;
    tsp3220_err = 0;
    if (tsp3220_device_closed (instrID))
        return tsp3220_err;

    Fmt(buffer,"I%d %f\012",opno,current);
    if (tsp3220_write_data(instrID,buffer,NumFmtdBytes()))
        return tsp3220_err;
    return tsp3220_err;  /* success */
}
/*=========================================================================*/
/* Function: Set OVP                                                       */
/* Purpose:  set the OVP value for the selected output                     */
/*=========================================================================*/
int tsp3220_set_ovp(instrID,opno,ovp)
int instrID;
int opno;
double ovp;
{
    if (tsp3220_invalid_integer_range (instrID, 1, tsp3220_MAX_INSTR, -1) != 0)
        return tsp3220_err;
    tsp3220_err = -2;       /* assume error */
    if (opno < 1)
        return tsp3220_err;
    if (tsp3220_model == 3222 && opno > 2)
        return tsp3220_err;
    tsp3220_err = 0;
    if (tsp3220_device_closed (instrID))
        return tsp3220_err;

    Fmt(buffer,"OVP%d %f\012",opno,ovp);
    if (tsp3220_write_data(instrID,buffer,NumFmtdBytes()))
        return tsp3220_err;
    return tsp3220_err;  /* success */
}
/*=========================================================================*/
/* Function: Set output status                                             */
/* Purpose:  Set the status for the selected output to ON or OFF           */
/*=========================================================================*/
int tsp3220_set_op_stat (instrID,opno,op_stat)
int instrID;
int opno;
int op_stat;
{
    if (tsp3220_invalid_integer_range (instrID, 1, tsp3220_MAX_INSTR, -1) != 0)
        return tsp3220_err;
    tsp3220_err = -2;       /* assume error */
    if (opno < 1)
        return tsp3220_err;
    if (tsp3220_model == 3222 && opno > 2)
        return tsp3220_err;
    tsp3220_err = 0;
    if (tsp3220_invalid_integer_range (op_stat, 0, 1, -3) != 0)
        return tsp3220_err;
    if (tsp3220_device_closed (instrID))
        return tsp3220_err;

    Fmt(buffer,"OP%d %d\012",opno,op_stat);
    if (tsp3220_write_data(instrID,buffer,NumFmtdBytes()))
        return tsp3220_err;
    return tsp3220_err;  /* success */
}
/*=========================================================================*/
/* Function: Set danping                                                   */
/* Purpose:  Set the readback meter damping for the seleced output to ON   */
/*           or OFF                                                        */
/*=========================================================================*/
int tsp3220_set_damping (instrID,opno,dmpg)
int instrID;
int opno;
int dmpg;
{
    if (tsp3220_invalid_integer_range (instrID, 1, tsp3220_MAX_INSTR, -1) != 0)
        return tsp3220_err;
    tsp3220_err = -2;       /* assume error */
    if (opno < 1)
        return tsp3220_err;
    if (tsp3220_model == 3222 && opno > 2)
        return tsp3220_err;
    tsp3220_err = 0;
    if (tsp3220_invalid_integer_range (dmpg, 0, 1, -3) != 0)
        return tsp3220_err;
    if (tsp3220_device_closed (instrID))
        return tsp3220_err;

    Fmt(buffer,"DAMPING%d %d\012",opno,dmpg);
    if (tsp3220_write_data(instrID,buffer,NumFmtdBytes()))
        return tsp3220_err;
    return tsp3220_err;  /* success */
}
/*=========================================================================*/
/* Function: Master output                                                 */
/* Purpose:  Set master output switch ON or OFF                            */
/*=========================================================================*/
int tsp3220_master_op (instrID,status)
int instrID;
int status;
{
    if (tsp3220_invalid_integer_range (instrID, 1, tsp3220_MAX_INSTR, -1) != 0)
        return tsp3220_err;
    if (tsp3220_device_closed (instrID))
        return tsp3220_err;

    Fmt(buffer,"MASTER %d\012",status);
    if (tsp3220_write_data(instrID,buffer,NumFmtdBytes()))
        return tsp3220_err;
    return tsp3220_err;  /* success */
}
/*=========================================================================*/
/* Function: Set status                                                    */
/* Purpose:  Set the value of a status enable register                     */
/*=========================================================================*/
int tsp3220_set_status (instrID,reg_type,byte_value)
int instrID;
int reg_type;
int byte_value;
{
char str[5];
    if (tsp3220_invalid_integer_range (instrID, 1, tsp3220_MAX_INSTR, -1) != 0)
        return tsp3220_err;
    if (tsp3220_invalid_integer_range (reg_type, 0, 3, -2) != 0)
        return tsp3220_err;
    if (tsp3220_invalid_integer_range (byte_value, 0, 255, -3) != 0)
        return tsp3220_err;
    if (tsp3220_device_closed (instrID))
        return tsp3220_err;

    switch (reg_type) {
    case 0:
        Fmt(str,"*SRE");
        break;
    case 1:
        Fmt(str,"*ESE");
        break;
    case 2:
        Fmt(str,"LSE");
        break;
    case 3:
        Fmt(str,"*PRE");
        break;
    }
    Fmt(buffer,"%s %d\012",str,byte_value);
    if (tsp3220_write_data(instrID,buffer,NumFmtdBytes()))
        return tsp3220_err;
    return tsp3220_err;  /* success */
}
/*=========================================================================*/
/* Function: Read Volts                                                    */
/* Purpose:  Read the voltage from the selected output                     */
/*=========================================================================*/
int tsp3220_read_op_volts (instrID,opno,voltage)
int instrID;
int opno;
double *voltage;
{
    if (tsp3220_invalid_integer_range (instrID, 1, tsp3220_MAX_INSTR, -1) != 0)
        return tsp3220_err;
    tsp3220_err = -2;       /* assume error */
    if (opno < 1)
        return tsp3220_err;
    if (tsp3220_model == 3222 && opno > 2)
        return tsp3220_err;
    if (tsp3220_device_closed (instrID))
        return tsp3220_err;

    Fmt(buffer,"V%dR?\012",opno);  /* read voltage */
    if (tsp3220_write_data(instrID,buffer,NumFmtdBytes()))
        return tsp3220_err;
    if (tsp3220_read_data(instrID,buffer,50))
        return tsp3220_err;
    Scan(buffer,"%fV",voltage); /* extract voltage value */
    return tsp3220_err;  /* success */
}
/*=========================================================================*/
/* Function: Read current                                                  */
/* Purpose:  Read the current from the selected output                     */
/*=========================================================================*/
int tsp3220_read_op_current(instrID,opno,current)
int instrID;
int opno;
double *current;
{
    if (tsp3220_invalid_integer_range (instrID, 1, tsp3220_MAX_INSTR, -1) != 0)
        return tsp3220_err;
    tsp3220_err = -2;       /* assume error */
    if (opno < 1)
        return tsp3220_err;
    if (tsp3220_model == 3222 && opno > 2)
        return tsp3220_err;
    if (tsp3220_device_closed (instrID))
        return tsp3220_err;

    Fmt(buffer,"I%dR?\012",opno);  /* read current */
    if (tsp3220_write_data(instrID,buffer,NumFmtdBytes()))
        return tsp3220_err;
    if (tsp3220_read_data(instrID,buffer,50))
        return tsp3220_err;
    Scan(buffer,"%fA",current); /* extract current value */
    return tsp3220_err;  /* success */
}
/*=========================================================================*/
/* Function: Read status                                                   */
/* Purpose:  Read and return the selected status register value            */
/*=========================================================================*/
int tsp3220_read_status (instrID,reg_type,value)
int instrID;
int reg_type;
int *value;
{
char str[7];
    if (tsp3220_invalid_integer_range (instrID, 1, tsp3220_MAX_INSTR, -1) != 0)
        return tsp3220_err;
    if (tsp3220_invalid_integer_range (reg_type, 0, 6, -2) != 0)
        return tsp3220_err;
    if (tsp3220_device_closed (instrID))
        return tsp3220_err;

    switch (reg_type) {
    case 0:
        Fmt(str,"*STB?\012");
        break;
    case 1:
        Fmt(str,"*ESR?\012");
        break;
    case 2:
        Fmt(str,"LSR?\012");
        break;
    case 3:
        Fmt(str,"*SRE?\012");
        break;
    case 4:
        Fmt(str,"*ESE?\012");
        break;
    case 5:
        Fmt(str,"LSE?\012");
        break;
    case 6:
        Fmt(str,"*PRE?\012");
        break;
    }
    if (tsp3220_write_data(instrID,str,NumFmtdBytes()))
        return tsp3220_err;
    if (tsp3220_read_data(instrID,buffer,50))
        return tsp3220_err;
    Scan(buffer,"%d",value); /* extract value */
    return tsp3220_err;  /* success */
}
/*=========================================================================*/
/* Function: Store                                                         */
/* Purpose:  Save settings to local non-volatile memory                    */
/*=========================================================================*/
int tsp3220_store (instrID,store_no)
int instrID;
int store_no;
{
    if (tsp3220_invalid_integer_range (instrID, 1, tsp3220_MAX_INSTR, -1) != 0)
        return tsp3220_err;
    if (tsp3220_invalid_integer_range (store_no, 1, 25, -2) != 0)
        return tsp3220_err;
    if (tsp3220_device_closed (instrID))
        return tsp3220_err;

    Fmt(buffer,"*SAV %d\012",store_no);
    if (tsp3220_write_data(instrID,buffer,NumFmtdBytes()))
        return tsp3220_err;
    return tsp3220_err;  /* success */
}
/*=========================================================================*/
/* Function: Recall                                                        */
/* Purpose:  Recall settings from local non-volatile memory                */
/*=========================================================================*/
int tsp3220_recall (instrID,store_no)
int instrID;
int store_no;
{
    if (tsp3220_invalid_integer_range (instrID, 1, tsp3220_MAX_INSTR, -1) != 0)
        return tsp3220_err;
    if (tsp3220_invalid_integer_range (store_no, 1, 25, -2) != 0)
        return tsp3220_err;
    if (tsp3220_device_closed (instrID))
        return tsp3220_err;

    Fmt(buffer,"*RCL %d\012",store_no);
    if (tsp3220_write_data(instrID,buffer,NumFmtdBytes()))
        return tsp3220_err;
    return tsp3220_err;  /* success */
}
/*=========================================================================*/
/* Function: Send command                                                  */
/* Purpose:  Send any command to the instrument                            */
/*=========================================================================*/
int tsp3220_send_cmd (instrID,cmd_buf)
int instrID;
char *cmd_buf;
{
char output_buf[2000];
    if (tsp3220_device_closed (instrID))
        return tsp3220_err;

    Fmt(output_buf,"%s",cmd_buf);
    if (tsp3220_write_data(instrID,cmd_buf,NumFmtdBytes()))
        return tsp3220_err;
    return tsp3220_err;  /* success */
}
/*=========================================================================*/
/* Function: Get response                                                  */
/* Purpose:  Get the response from a query sent by send command            */
/*=========================================================================*/
int tsp3220_get_response (instrID,cmd_buf,count)
int instrID;
char *cmd_buf;
int *count;
{
    if (tsp3220_device_closed (instrID))
        return tsp3220_err;

    if (tsp3220_read_data(instrID,cmd_buf,2000))
        return tsp3220_err;
    *count = ibcnt;
    return tsp3220_err;
}
/*=========================================================================*/
/* Function: Send file                                                     */
/* Purpose:  Send data from a disk file to the instrument                  */
/*=========================================================================*/
int tsp3220_send_from_file (instrID,fname)
int instrID;
char *fname;
{
    if (tsp3220_device_closed (instrID))
        return tsp3220_err;

    if (tsp3220_write_data_file(instrID,fname))
        return tsp3220_err;
    return tsp3220_err;  /* success */
}
/*=========================================================================*/
/* Function: Get to file                                                   */
/* Purpose:  Get data from the instrument to a disk file                   */
/*=========================================================================*/
int tsp3220_get_to_file (instrID,fname)
int instrID;
char *fname;
{
    if (tsp3220_device_closed (instrID))
        return tsp3220_err;

    if (tsp3220_read_data_file(instrID,fname))
        return tsp3220_err;
    return tsp3220_err;  /* success */
}
/*=========================================================================*/
/* Function: Close                                                         */
/* Purpose:  This function closes the instrument.                          */
/*=========================================================================*/
int tsp3220_close (instrID)
int instrID;
{
    if (tsp3220_invalid_integer_range (instrID, 1, tsp3220_MAX_INSTR, -1) != 0)
        return tsp3220_err;

    if (tsp3220_device_closed (instrID))
        return tsp3220_err;

    tsp3220_close_instr (instrID);
    return tsp3220_err;
}
/*= UTILITY ROUTINES ======================================================*/

/*=========================================================================*/
/* Function: Open Instrument                                               */
/* Purpose:  This function locates and initializes an entry in the         */
/*           Instrument Table and the GPIB device table for the            */
/*           instrument.  If successful, the instrument ID is returned,    */
/*           else a -1 is returned.  The size of the Instrument Table can  */
/*           be changed in the include file by altering the constant       */
/*           tsp3220_MAX_INSTR.                                               */
/*=========================================================================*/
int tsp3220_open_instr (addr)
int addr;
{
    int i, instrID;

    instrID = 0;

/* Check to see if the instrument is already in the Instrument Table. */

    for (i=1; i<= tsp3220_MAX_INSTR; i++)
        if (address[i] == addr)
            {
            instrID = i;
            i = tsp3220_MAX_INSTR;
            }

/* If it is not in the instrument table, open an entry for the instrument. */

    if (instrID <= 0)
        for (i=1; i<= tsp3220_MAX_INSTR; i++)
            if (address[i] == 0)
                {
                instrID = i;
                i = tsp3220_MAX_INSTR;
                }

/* If an entry could not be opened in the Instrument Table, return an error.*/

    if (instrID <= 0)
        {
        tsp3220_err = 220;
        return -1;
        }

/*  If the device has not been opened in the GPIB device table (bd[ID] = 0),*/
/*  then open it.                                                           */

    if (bd[instrID] <= 0)
        {
        if (instr_cnt <= 0)
            CloseInstrDevs("TPLP");
        bd[instrID] = OpenDev ("", "TPLP");
        if (bd[instrID] <= 0)
            {
            tsp3220_err = 220;
            return -1;
            }
        instr_cnt += 1;
        address[instrID] = addr;
        }

/*  Change the primary address of the device    */

    if (ibpad (bd[instrID], addr) < 0)
        {
        tsp3220_err = 233;
        return -1;
        }

    return instrID;
}
/*=========================================================================*/
/* Function: Close Instrument                                              */
/* Purpose:  This function closes the instrument by removing it from the   */
/*           GPIB device table and setting the address and the bd to zero  */
/*           in the Instrument Table.  The return value is equal to the    */
/*           global error variable.                                        */
/*=========================================================================*/
int tsp3220_close_instr (instrID)
int instrID;
{
    if (bd[instrID] != 0)
        {
        CloseDev (bd[instrID]);
        bd[instrID] = 0;
        address[instrID] = 0;
        instr_cnt -= 1;
        }
    else
        tsp3220_err = 221;

    return tsp3220_err;
}
/*=========================================================================*/
/* Function: Invalid Integer Range                                         */
/* Purpose:  This function checks an integer to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int tsp3220_invalid_integer_range (val, min, max, err_code)
int val;
int min;
int max;
int err_code;
{
  if ((val < min) || (val > max))
    {
    tsp3220_err = err_code;
    return -1;
    }
  return 0;
}
/*=========================================================================*/
/* Function: Device Closed                                                 */
/* Purpose:  This function checks to see if the module has been            */
/*           initialized.  If the device has not been opened, a 1 is       */
/*           returned, 0 otherwise.                                        */
/*=========================================================================*/
int tsp3220_device_closed (instrID)
int instrID;
{
    if (bd[instrID] <= 0)
        {
        tsp3220_err = 232;
        return -1;
        }
    return 0;
}
/*=========================================================================*/
/* Function: Read Data                                                     */
/* Purpose:  This function reads a buffer of data from the instrument. The */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
int tsp3220_read_data (instrID, buf, cnt)
int instrID;
char *buf;
int cnt;
{
    if (ibrd(bd[instrID], buf, (long)cnt) <= 0)
        tsp3220_err = 231;
    else
        tsp3220_err = 0;

    return tsp3220_err;
}
/*=========================================================================*/
/* Function: Write Data                                                    */
/* Purpose:  This function writes a buffer of data to the instrument. The  */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
int tsp3220_write_data (instrID, buf, cnt)
int instrID;
char *buf;
int cnt;
{
    if (ibwrt(bd[instrID], buf, (long)cnt) <= 0)
        tsp3220_err = 230;
    else
        tsp3220_err = 0;

    return tsp3220_err;
}
/*=========================================================================*/
/* Function: Read Data File                                                */
/* Purpose:  This function reads a buffer of data from the instrument and  */
/*           stores it to the file specified by "filename".  Filename must */
/*           either be a string, such as "C:\lw\instr\file" or a pointer   */
/*           to such a string.  The return value is equal to the global    */
/*           error variable.                                               */
/*=========================================================================*/
int tsp3220_read_data_file (instrID, filename)
int instrID;
char *filename;
{
    if (ibrdf (bd[instrID], filename) <= 0)
        tsp3220_err = 229;
    else
        tsp3220_err = 0;

    return tsp3220_err;
}
/*=========================================================================*/
/* Function: Write Data File                                               */
/* Purpose:  This function writes a buffer of data from the file specified */
/*           by "filename" to the instrument. Filename must either be a    */
/*           string, such as "C:\lw\instr\file" or a pointer to such a     */
/*           string.  The return value is equal to the global error        */
/*           variable.                                                     */
/*=========================================================================*/
int tsp3220_write_data_file (instrID, filename)
int instrID;
char *filename;
{
    if (ibwrtf (bd[instrID], filename) <= 0)
        tsp3220_err = 228;
    else
        tsp3220_err = 0;

    return tsp3220_err;
}
/*= THE END ===============================================================*/
